﻿using Hims.Api.Models;
using Hims.Api.Utilities;
using Hims.Domain.Services;
using Hims.Shared.DataFilters;
using Hims.Shared.EntityModels;
using Hims.Shared.Library.Enums;
using Hims.Shared.UserModels;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;

namespace Hims.Api.Controllers
{

    /// <inheritdoc />
    [Authorize]
    [Route("api/machine")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class MachineController : BaseController
    {
        /// <summary>
        /// The idproof services.
        /// </summary>
        private readonly IMachineService machineService;

        /// <summary>
        /// The pharmacy service.
        /// </summary>
        private readonly ILabLogService labLogService;
        /// <inheritdoc />
        public MachineController(IMachineService machineService, ILabLogService labLogService)
        {
            this.machineService = machineService;
            this.labLogService = labLogService;
        }

        /// <summary>
        /// Inserts the hoc asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="location">The location.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("add")]
        public async Task<ActionResult> InsertAsync([FromBody] MachineModel model, [FromHeader] LocationHeader location)
        {


            model = (MachineModel)EmptyFilter.Handler(model);
            var response = await this.machineService.InsertAsync(model);
            var labLogModel = new LabLogModel
            {
                AccountId = model.CreatedBy,
                LabLogTypeId = (int)LabLogTypes.Lab_Masters_Machine,
                LogFrom = (short)model.LoginRoleId,
                LogDate = DateTime.Now,
                LogDescription = $"{model.CreatedByName} has added <b>Machine</b> of <strong>{model.MachineName}</strong> successfully.",
                LocationId = Convert.ToInt32(location.LocationId)
            };
            await this.labLogService.LogAsync(labLogModel);
            return this.Success(response);


        }

        /// <summary>
        /// Inserts the hoc asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="location">The location.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("update")]
        public async Task<ActionResult> UpdateAsync([FromBody] MachineModel model, [FromHeader] LocationHeader location)
        {
            model = (MachineModel)EmptyFilter.Handler(model);
            var response = await this.machineService.UpdateAsync(model);
            var labLogModel = new LabLogModel
            {
                AccountId = model.CreatedBy,
                LabLogTypeId = (int)LabLogTypes.Lab_Masters_Machine,
                LogFrom = (short)model.LoginRoleId,
                LogDate = DateTime.Now,
                LogDescription = $"{model.CreatedByName} has Updated <b>Machine</b> of <strong>{model.MachineName}</strong> successfully.",
                LocationId = Convert.ToInt32(location.LocationId)
            };
            await this.labLogService.LogAsync(labLogModel);
            return this.Success(response);
        }
        /// <summary>
        /// Fetches all asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="location">The location.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("fetch-all")]
        public async Task<ActionResult> FetchAsync([FromBody] MachineModel model)
        {
            model = (MachineModel)EmptyFilter.Handler(model);
            var response = await this.machineService.FetchAllAsync(model);
            return this.Success(response);
        }

        /// <summary>
        ///modifies all asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="location">The location.</param>
        /// <returns></returns>
       

        [HttpPost]
        [Route("modify-status")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ModifyStatusAsync([FromBody] MachineModel model, [FromHeader] LocationHeader location)
        {
            model = (MachineModel)EmptyFilter.Handler(model);
            var response = await this.machineService.ActivateOrDeactivateMachine(model);
            if (response == 0)
            {
                return this.ServerError();
            }
            var labLogModel = new LabLogModel
            {
                AccountId = model.CreatedBy,
                LabLogTypeId = (int)LabLogTypes.Lab_Masters_Machine,
                LogFrom = (short)model.LoginRoleId,
                LogDate = DateTime.Now,
                LogDescription = $" { model.CreatedByName } has changed status of <b>Machine</b> - <strong>{ model.MachineName }</strong> on { DateTime.Now }",
                LocationId = Convert.ToInt32(location.LocationId)
            };
            await this.labLogService.LogAsync(labLogModel);



            return this.Success("Machine has been modified status successfully.");

        }

    }
    }